/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package chart;

import java.awt.Color;

import com.inet.report.Chart2;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.RDC;
import com.inet.report.ReportException;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;
import com.inet.report.chart.ChartTitle;
import com.inet.report.chart.Legend;
import com.inet.report.chart.LegendPlacement;
import com.inet.report.chart.axis.ContinuousNumberAxis;
import com.inet.report.chart.axis.GroupAxis;
import com.inet.report.chart.dataset.OneGroupDataset;
import com.inet.report.chart.format.FixedNumberFormat;
import com.inet.report.chart.plot.BarPlot;
import com.inet.report.chart.plot.BarStyle;
import com.inet.report.chart.plot.StandardPlot;

import rdc.RDCSample;

/**
 * This sample demonstrates how you can create a bar chart and use this properties.
 */
public class BarSample extends RDCSample {

    /**
     * The category column name.
     */
    private static String category = "Category";

    /**
     * The value column name.
     */
    private static String value    = "Value";

    /**
     * Creates an empty engine, adds a chart and sets the data.
     * @param exportFmt the export format set by super class.
     * @return returns the created engine.
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //Creates a new empty engine.
            Engine engine = RDC.createEmptyEngine( exportFmt );

            //Creates two database columns used by chart.
            TableSource tableSource = engine.getDatabaseTables().getDatasource( 0 ).createTableSource( "sample" );
            tableSource.addColumn( category, Field.STRING );
            tableSource.addColumn( value, Field.NUMBER );

            //Gets the first section of report header.
            Section section = engine.getArea( Engine.REPORT_HEADER ).getSection( 0 );

            //Creates a new bar chart in the section.
            Chart2 barChart = section.addChart2( BarStyle.BAR3D, 0, 0, 8000, 6000 );

            //Gets the legend of chart to set this properties.
            Legend legend = barChart.getLegend();
            //Moves the legend to the bottom site of chart.
            legend.setLegendPlacement( LegendPlacement.BOTTOM );
            //Sets the legend label color.
            legend.setLegendColor( RDC.COLOR_WHITE );
            //Sets the legend background color.
            legend.setBackColor( RDC.COLOR_GRAY );

            //Gets the header title of chart.
            ChartTitle title = barChart.getHeaderTitle();
            //Sets the custom label instead of automatic generated label.
            title.setShowAutoTitle( false );
            title.setTitle( "Header title" );
            //Sets the title label color.
            title.setColor( RDC.toCcColor( new Color( 85, 85, 85 ) ) );

            //Creates a data source of chart.
            BarPlot plot = (BarPlot)barChart.getPlot();
            //This chart gets a one group dataset. 
            OneGroupDataset dataset = new OneGroupDataset( barChart );
            //Adds the category field as group.
            DatabaseField categoryField = tableSource.getDatabaseField( category );
            dataset.setCategoryGroup( categoryField );
            //Adds the value field. The chart should shows a sum of values for each category.
            DatabaseField valueField = tableSource.getDatabaseField( value );
            dataset.addDataField( SummaryField.SUM, valueField, null, 0 );
            plot.setDataset( dataset );

            //Sets the plot background color.
            plot.setBackColor( RDC.COLOR_GRAY );
            //Sets the chart items outline.
            plot.setOutlineColor( RDC.toCcColor( new Color( 150, 0, 0 ) ) );

            //Sets the labels item value will be shown.
            plot.setShowValue( true );
            //Sets the item label color.
            plot.setItemLabelColor( RDC.COLOR_WHITE );
            //Sets the item label position.
            plot.setItemLabelPosition( StandardPlot.ITEM_LABEL_POSITION_CENTER );

            //Gets the category axis to change it properties.
            GroupAxis categoryAxis = (GroupAxis)plot.getCategoryAxis();
            //Rotates the axis tick labels on 45°
            categoryAxis.setAutoFitTickLabel( false );
            categoryAxis.setTickLabelRotationAngle( Math.PI / 4.0 );
            //Sets a empty axis label.
            categoryAxis.getTitle().setShowAutoTitle( false );
            categoryAxis.getTitle().setTitle( "" );

            //Gets the data axis to change it properties.
            ContinuousNumberAxis valueAxis = (ContinuousNumberAxis)plot.getDataAxis();
            //Sets the number of divisions.
            valueAxis.setNumberOfDivisions( Integer.valueOf( 5 ) );
            //Sets a custom number format with two decimal places.
            FixedNumberFormat format = new FixedNumberFormat( FixedNumberFormat.FORMAT_NUMBER );
            format.setNumberOfDecimalPlaces( 2 );
            valueAxis.setTickLabelFormat( format );
            //Sets a empty axis label.
            valueAxis.getTitle().setShowAutoTitle( false );
            valueAxis.getTitle().setTitle( "" );

            //Sets the report data. 
            setData( engine );
            return engine;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Sets the report data. This sets the two data columns used by chart.
     * @param engine the current engine
     * @throws ReportException if engine is not initialized or finished.
     */
    private static void setData( Engine engine ) throws ReportException {
        String[] columns = { category, value };
        Object[][] data = new Object[4][];
        data[0] = new Object[2];
        data[0][0] = "Category 1";
        data[0][1] = Integer.valueOf( 1 );

        data[1] = new Object[2];
        data[1][0] = "Category 2";
        data[1][1] = Integer.valueOf( 2 );

        data[2] = new Object[2];
        data[2][0] = "Category 3";
        data[2][1] = Integer.valueOf( 5 );

        data[3] = new Object[2];
        data[3][0] = "Category 4";
        data[3][1] = Integer.valueOf( 4 );

        engine.setData( columns, data );
    }

    /**
     * Starts this sample.
     * @param argv nothing
     */
    public static void main( String[] argv ) {
        new BarSample().initUI();
    }
}
